<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class PortfolioController extends Controller
{
    public function index()
    {
        $companies = DB::table('portfolio_companies')
            ->leftJoin('portfolio_categories', 'portfolio_companies.portfolio_category_id', '=', 'portfolio_categories.id')
            ->select('portfolio_companies.*', 'portfolio_categories.name as category_name')
            ->orderBy('sort_order')
            ->get();
        return view('admin.portfolio.index', compact('companies'));
    }

    public function create()
    {
        $categories = DB::table('portfolio_categories')->orderBy('name')->get();
        return view('admin.portfolio.create', compact('categories'));
    }

    public function store(Request $request)
    {
        $request->validate(['name' => 'required', 'portfolio_category_id' => 'required']);

        $data = $request->except('_token', 'logo_file', 'featured_image_file');
        $data['slug'] = Str::slug($request->name);
        $data['created_at'] = now();
        $data['updated_at'] = now();

        if ($request->hasFile('logo_file')) {
            $data['logo'] = '/storage/' . $request->file('logo_file')->store('uploads/portfolio', 'public');
        } elseif ($request->filled('logo')) {
            $data['logo'] = $request->input('logo');
        }

        if ($request->hasFile('featured_image_file')) {
            $data['featured_image'] = '/storage/' . $request->file('featured_image_file')->store('uploads/portfolio', 'public');
        } elseif ($request->filled('featured_image')) {
            $data['featured_image'] = $request->input('featured_image');
        }
        // Handle services array (simple comma separated to json or multiple inputs)
        // For simplicity, let's assume textarea one per line
        if ($request->services) {
            $data['services'] = json_encode(array_filter(array_map('trim', explode("\n", $request->services))));
        }

        DB::table('portfolio_companies')->insert($data);

        return redirect()->route('admin.portfolio.index')->with('success', 'Company added.');
    }

    public function edit($id)
    {
        $company = DB::table('portfolio_companies')->find($id);
        $company->services = implode("\n", json_decode($company->services ?? '[]'));
        $categories = DB::table('portfolio_categories')->orderBy('name')->get();
        return view('admin.portfolio.edit', compact('company', 'categories'));
    }

    public function update(Request $request, $id)
    {
        $request->validate(['name' => 'required', 'portfolio_category_id' => 'required']);

        $data = $request->except('_token', '_method', 'logo_file', 'featured_image_file');
        // Slug update? Only if needed, strictly safe to keep old slug or update if name changed drastically? user preference. 
        // I'll update slug if name changed.
        $data['slug'] = Str::slug($request->name);
        $data['updated_at'] = now();

        if ($request->hasFile('logo_file')) {
            $data['logo'] = '/storage/' . $request->file('logo_file')->store('uploads/portfolio', 'public');
        } elseif ($request->filled('logo')) {
            $data['logo'] = $request->input('logo');
        }

        if ($request->hasFile('featured_image_file')) {
            $data['featured_image'] = '/storage/' . $request->file('featured_image_file')->store('uploads/portfolio', 'public');
        } elseif ($request->filled('featured_image')) {
            $data['featured_image'] = $request->input('featured_image');
        }
        if ($request->services) {
            $data['services'] = json_encode(array_filter(array_map('trim', explode("\n", $request->services))));
        }

        DB::table('portfolio_companies')->where('id', $id)->update($data);

        return redirect()->route('admin.portfolio.index')->with('success', 'Company updated.');
    }

    public function destroy($id)
    {
        DB::table('portfolio_companies')->delete($id);
        return redirect()->route('admin.portfolio.index')->with('success', 'Company deleted.');
    }
}
