<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Menu;
use App\Models\MenuItem;
use App\Models\Page;

class MenuController extends Controller
{
    public function index()
    {
        $menus = Menu::withCount('items')->get();
        return view('admin.menus.index', compact('menus'));
    }

    public function store(Request $request)
    {
        $request->validate(['name' => 'required|string', 'location' => 'nullable|string|unique:menus,location']);
        Menu::create($request->only('name', 'location'));
        return redirect()->back()->with('success', 'Menu created successfully.');
    }

    public function builder(Menu $menu)
    {
        $menu->load('rootItems.children'); // Eager load tree
        $pages = Page::select('id', 'title', 'slug')->get(); // For "Add Page" list
        return view('admin.menus.builder', compact('menu', 'pages'));
    }

    // AJAX: Add Item
    public function storeItem(Request $request, Menu $menu)
    {
        $request->validate([
            'title' => 'requried_without:page_id|string|nullable',
            'page_id' => 'nullable|exists:pages,id',
            'url' => 'nullable|string',
            'target' => 'nullable|string',
        ]);

        $item = $menu->items()->create([
            'title' => $request->title,
            'page_id' => $request->page_id,
            'url' => $request->url,
            'target' => $request->target ?? '_self',
            'order' => 999
        ]);

        return response()->json(['status' => 'success', 'item' => $item]);
    }

    // AJAX: Update Structure (Reorder/Nest)
    public function updateStructure(Request $request, Menu $menu)
    {
        $structure = $request->input('structure'); // JSON array of {id, children: []}
        $this->processStructure($structure);
        return response()->json(['status' => 'success']);
    }

    private function processStructure($items, $parentId = null)
    {
        foreach ($items as $index => $itemData) {
            $item = MenuItem::find($itemData['id']);
            if ($item) {
                $item->update(['parent_id' => $parentId, 'order' => $index]);
                if (isset($itemData['children']) && !empty($itemData['children'])) {
                    $this->processStructure($itemData['children'], $item->id);
                }
            }
        }
    }

    // AJAX: Update Item Details
    public function updateItem(Request $request, $itemId)
    {
        $item = MenuItem::findOrFail($itemId);
        $item->update($request->only('title', 'url', 'target', 'css_class', 'icon'));
        return response()->json(['status' => 'success']);
    }

    // AJAX: Delete Item
    public function destroyItem($itemId)
    {
        $item = MenuItem::findOrFail($itemId);
        $item->delete();
        return response()->json(['status' => 'success']);
    }

    public function destroy(Menu $menu)
    {
        $menu->delete();
        return redirect()->route('admin.menus.index')->with('success', 'Menu deleted.');
    }
}
