<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\File;

class MediaController extends Controller
{
    public function index(Request $request)
    {
        $files = [];
        $directories = [
            'uploads',
            'uploads/settings',
            'uploads/sections',
            'uploads/portfolio',
            'uploads/team',
            'uploads/news',
            'uploads/videos', // Added for video management
            'img',
            'img/demos/business-consulting-2',
            'img/demos/business-consulting-2/blog'
        ];

        foreach ($directories as $dir) {
            $path = public_path($dir);
            if (File::exists($path)) {
                $filesInDir = File::files($path);
                foreach ($filesInDir as $file) {
                    $files[] = [
                        'name' => $file->getFilename(),
                        'path' => $dir . '/' . $file->getFilename(),
                        'url' => asset($dir . '/' . $file->getFilename()),
                        'size' => $this->formatSize($file->getSize()),
                        'last_modified' => date('Y-m-d H:i:s', $file->getMTime()),
                        'is_image' => in_array(strtolower($file->getExtension()), ['jpg', 'jpeg', 'png', 'gif', 'webp', 'svg']),
                        'extension' => $file->getExtension()
                    ];
                }
            }
        }

        // Sort by newest first
        usort($files, function ($a, $b) {
            return $b['last_modified'] <=> $a['last_modified'];
        });

        if ($request->wantsJson()) {
            return response()->json($files);
        }

        return view('admin.media.index', compact('files'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'file' => 'required|file|max:10240', // 10MB max
            'folder' => 'nullable|string'
        ]);

        $folder = $request->input('folder', 'uploads');
        // Sanitize folder path to prevent traversal/root access
        if (!in_array($folder, ['uploads', 'uploads/settings', 'uploads/sections', 'uploads/portfolio', 'uploads/team', 'uploads/news', 'uploads/videos'])) {
            $folder = 'uploads';
        }

        if ($request->hasFile('file')) {
            $file = $request->file('file');
            $filename = time() . '_' . $file->getClientOriginalName();
            $file->move(public_path($folder), $filename);

            if ($request->wantsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => 'File uploaded successfully.',
                    'file' => [
                        'name' => $filename,
                        'path' => $folder . '/' . $filename,
                        'url' => asset($folder . '/' . $filename),
                        'is_image' => in_array(strtolower($file->getClientOriginalExtension()), ['jpg', 'jpeg', 'png', 'gif', 'webp', 'svg'])
                    ]
                ]);
            }

            return redirect()->route('admin.media.index')->with('success', 'File uploaded successfully.');
        }

        if ($request->wantsJson()) {
            return response()->json(['success' => false, 'message' => 'No file uploaded.'], 422);
        }

        return redirect()->back()->with('error', 'No file uploaded.');
    }

    public function destroy(Request $request)
    {
        $path = $request->input('path');

        // Security check: ensure path starts with uploads/
        if (!str_starts_with($path, 'uploads/')) {
            if ($request->wantsJson()) {
                return response()->json(['success' => false, 'message' => 'Invalid file path. Cannot delete system or demo files.'], 403);
            }
            return redirect()->back()->with('error', 'Invalid file path.');
        }

        if (File::exists(public_path($path))) {
            File::delete(public_path($path));

            if ($request->wantsJson()) {
                return response()->json(['success' => true, 'message' => 'File deleted successfully.']);
            }

            return redirect()->route('admin.media.index')->with('success', 'File deleted successfully.');
        }

        if ($request->wantsJson()) {
            return response()->json(['success' => false, 'message' => 'File not found.'], 404);
        }

        return redirect()->back()->with('error', 'File not found.');
    }

    public function rename(Request $request)
    {
        $request->validate([
            'path' => 'required|string',
            'new_name' => 'required|string|regex:/^[a-zA-Z0-9_\-\.]+$/'
        ]);

        $path = $request->input('path');
        $newName = $request->input('new_name');

        // Security check: ensure path starts with uploads/
        if (!str_starts_with($path, 'uploads/')) {
            return response()->json(['success' => false, 'message' => 'Invalid file path. Cannot rename system or demo files.'], 403);
        }

        $oldPath = public_path($path);
        if (!File::exists($oldPath)) {
            return response()->json(['success' => false, 'message' => 'File not found.'], 404);
        }

        $directory = dirname($oldPath);
        $extension = File::extension($oldPath);

        // Ensure new name has extension
        if (!str_ends_with($newName, '.' . $extension)) {
            $newName .= '.' . $extension;
        }

        $newPath = $directory . '/' . $newName;

        if (File::exists($newPath)) {
            return response()->json(['success' => false, 'message' => 'A file with this name already exists.'], 422);
        }

        File::move($oldPath, $newPath);

        // Calculate new relative path for frontend
        $newRelativePath = dirname($path) . '/' . $newName;

        return response()->json([
            'success' => true,
            'message' => 'File renamed successfully.',
            'new_path' => $newRelativePath,
            'new_url' => asset($newRelativePath),
            'new_name' => $newName
        ]);
    }

    public function uploadVideo(Request $request)
    {
        $request->validate([
            'video' => 'required|file|mimetypes:video/mp4,video/webm,video/ogg,video/quicktime|max:512000', // 500MB max
        ]);

        if ($request->hasFile('video')) {
            $file = $request->file('video');
            $filename = time() . '_' . $file->getClientOriginalName();

            // Ensure directory exists
            if (!File::exists(public_path('uploads/videos'))) {
                File::makeDirectory(public_path('uploads/videos'), 0755, true);
            }

            $file->move(public_path('uploads/videos'), $filename);

            return response()->json([
                'success' => true,
                'url' => asset('uploads/videos/' . $filename),
                'filename' => $filename
            ]);
        }

        return response()->json(['success' => false, 'message' => 'No file uploaded'], 400);
    }

    private function formatSize($bytes)
    {
        if ($bytes >= 1073741824) {
            $bytes = number_format($bytes / 1073741824, 2) . ' GB';
        } elseif ($bytes >= 1048576) {
            $bytes = number_format($bytes / 1048576, 2) . ' MB';
        } elseif ($bytes >= 1024) {
            $bytes = number_format($bytes / 1024, 2) . ' KB';
        } elseif ($bytes > 1) {
            $bytes = $bytes . ' bytes';
        } elseif ($bytes == 1) {
            $bytes = $bytes . ' byte';
        } else {
            $bytes = '0 bytes';
        }

        return $bytes;
    }
}
